<?php
/* --------------------------------------------------------------
 ConfigurationRepository.php 2020-03-09
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2020 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

declare(strict_types=1);

namespace Gambio\Core\Configuration\Repositories\Implementations;

use Gambio\Core\Configuration\Models\Read;
use Gambio\Core\Configuration\Models\Write;
use Gambio\Core\Configuration\Repositories\Components\ConfigurationFactory;
use Gambio\Core\Configuration\Repositories\Components\ConfigurationReader;
use Gambio\Core\Configuration\Repositories\Components\ConfigurationWriter;
use Gambio\Core\Configuration\Repositories\ConfigurationRepository as ConfigurationRepositoryInterface;

/**
 * Class ConfigurationRepository
 * @package Gambio\Core\Configuration\Repositories\Implementations
 */
class ConfigurationRepository implements ConfigurationRepositoryInterface
{
    /**
     * @var ConfigurationReader
     */
    private $reader;
    
    /**
     * @var ConfigurationWriter
     */
    private $writer;
    
    /**
     * @var ConfigurationFactory
     */
    private $factory;
    
    
    /**
     * ConfigurationRepository constructor.
     *
     * @param ConfigurationReader  $reader
     * @param ConfigurationWriter  $writer
     * @param ConfigurationFactory $factory
     */
    public function __construct(ConfigurationReader $reader, ConfigurationWriter $writer, ConfigurationFactory $factory)
    {
        $this->reader  = $reader;
        $this->writer  = $writer;
        $this->factory = $factory;
    }
    
    
    /**
     * @inheritDoc
     */
    public function findWithLanguageCode(string $key, string $languageCode): ?Read\Interfaces\Configuration
    {
        $data = $this->reader->findWithLanguageCode($key, $languageCode);
        if (null === $data) {
            return null;
        }
        
        return $this->factory->createReadConfiguration($data);
    }
    
    
    /**
     * @inheritDoc
     */
    public function findWithLanguageId(string $key, int $languageId): ?Read\Interfaces\Configuration
    {
        $data = $this->reader->findWithLanguageId($key, $languageId);
        if (null === $data) {
            return null;
        }
        
        return $this->factory->createReadConfiguration($data);
    }
    
    
    /**
     * @inheritDoc
     */
    public function find(string $key): ?Read\Interfaces\Configuration
    {
        $data = $this->reader->find($key);
        if (null === $data) {
            return null;
        }
        
        return $this->factory->createReadConfiguration($data);
    }
    
    
    /**
     * @inheritDoc
     */
    public function has(string $key): bool
    {
        return $this->reader->has($key);
    }
    
    
    /**
     * @inheritDoc
     */
    public function update(Write\Interfaces\Configuration $configuration): void
    {
        $this->writer->update($configuration);
    }
    
    
    /**
     * @inheritDoc
     */
    public function add(Write\Interfaces\Configuration $configuration): void
    {
        $this->writer->add($configuration);
    }
    
    
    /**
     * @inheritDoc
     */
    public function delete(string ...$keys): void
    {
        $this->writer->delete(...$keys);
    }
}